<?php
declare(strict_types=1);

require_once __DIR__ . '/common.php';
require_once __DIR__ . '/token.php';

try {
    api_require_https();
    api_require_post();
    api_rate_limit('validate:' . gmap_client_ip(), 180, 60);

    $body = api_get_json_body();
    $token = trim((string)($body['token'] ?? ''));
    $hwidHash = api_validate_hwid_hash((string)($body['hwid_hash'] ?? ''));
    $appVersion = substr(trim((string)($body['app_version'] ?? 'unknown')), 0, 64);

    if ($token === '') {
        api_license_error('TOKEN_INVALID');
    }

    $payload = gmap_verify_token($token, gmap_hmac_secret());
    if (!is_array($payload)) {
        api_license_error('TOKEN_INVALID');
    }

    $tokenLicenseId = (int)($payload['license_id'] ?? 0);
    $tokenLicenseKeyHash = (string)($payload['license_key_hash'] ?? '');
    $tokenHwid = strtolower(trim((string)($payload['hwid_hash'] ?? '')));
    $tokenExpiresAt = (int)($payload['expires_at'] ?? 0);

    if ($tokenLicenseId <= 0 || $tokenLicenseKeyHash === '' || $tokenHwid === '') {
        api_license_error('TOKEN_INVALID');
    }
    if ($tokenExpiresAt > 0 && $tokenExpiresAt < time()) {
        api_license_error('TOKEN_EXPIRED');
    }
    if (!hash_equals($tokenHwid, $hwidHash)) {
        api_license_error('KEY_IN_USE');
    }

    $pdo = gmap_pdo();
    $now = gmap_now_utc();

    $pdo->beginTransaction();
    $stmt = $pdo->prepare('SELECT * FROM licenses WHERE id = :id LIMIT 1 FOR UPDATE');
    $stmt->execute([':id' => $tokenLicenseId]);
    $license = $stmt->fetch();

    if (!$license) {
        gmap_insert_log($pdo, null, 'deny', $hwidHash, 'INVALID_KEY license_id=' . $tokenLicenseId);
        $pdo->commit();
        api_license_error('INVALID_KEY');
    }

    $licenseId = (int)$license['id'];
    $status = (string)$license['status'];
    $expiresAt = (string)$license['expires_at'];
    $expiresTs = strtotime($expiresAt . ' UTC');
    $normalizedKey = gmap_normalize_license_key((string)$license['license_key']);
    $expectedHash = gmap_hash_license_key($normalizedKey);

    if (!hash_equals($expectedHash, $tokenLicenseKeyHash)) {
        gmap_insert_log($pdo, $licenseId, 'deny', $hwidHash, 'TOKEN_INVALID key hash mismatch v=' . $appVersion);
        $pdo->commit();
        api_license_error('TOKEN_INVALID');
    }
    if ($status === 'revoked') {
        gmap_insert_log($pdo, $licenseId, 'revoke', $hwidHash, 'KEY_REVOKED v=' . $appVersion);
        $pdo->commit();
        api_license_error('KEY_REVOKED');
    }
    if ($status === 'paused') {
        gmap_insert_log($pdo, $licenseId, 'pause', $hwidHash, 'KEY_PAUSED v=' . $appVersion);
        $pdo->commit();
        api_license_error('KEY_PAUSED');
    }
    if ($expiresTs !== false && $expiresTs < time()) {
        gmap_insert_log($pdo, $licenseId, 'deny', $hwidHash, 'KEY_EXPIRED v=' . $appVersion);
        $pdo->commit();
        api_license_error('KEY_EXPIRED');
    }

    $boundHwid = strtolower(trim((string)($license['bound_hwid'] ?? '')));
    if ($boundHwid === '') {
        gmap_insert_log($pdo, $licenseId, 'deny', $hwidHash, 'KEY_RESET_REQUIRED v=' . $appVersion);
        $pdo->commit();
        api_license_error('KEY_RESET_REQUIRED');
    }
    if (!hash_equals($boundHwid, $hwidHash)) {
        gmap_insert_log($pdo, $licenseId, 'deny', $hwidHash, 'KEY_IN_USE v=' . $appVersion);
        $pdo->commit();
        api_license_error('KEY_IN_USE');
    }

    $touch = $pdo->prepare('UPDATE licenses SET last_seen_at = :last_seen_at, last_ip = :last_ip WHERE id = :id');
    $touch->execute([
        ':last_seen_at' => $now,
        ':last_ip' => gmap_client_ip(),
        ':id' => $licenseId,
    ]);

    gmap_insert_log($pdo, $licenseId, 'validate', $hwidHash, 'OK v=' . $appVersion);

    $token = gmap_issue_license_token($license, $hwidHash, gmap_hmac_secret());
    $pdo->commit();

    api_respond([
        'ok' => true,
        'token' => $token,
        'expires_at' => $expiresAt,
        'status' => $status,
    ]);
} catch (Throwable $e) {
    if (isset($pdo) && $pdo instanceof PDO && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    api_respond([
        'ok' => false,
        'error' => 'SERVER_ERROR',
        'message' => 'Validation failed.',
    ], 500);
}
