<?php
declare(strict_types=1);

require_once __DIR__ . '/db.php';

$corsAllowOrigin = '*';
try {
    $corsAllowOrigin = gmap_cors_allow_origin();
} catch (Throwable $e) {
    $corsAllowOrigin = '*';
}

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: ' . $corsAllowOrigin);
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-DEBUG-KEY');
header('Vary: Origin');

function api_respond(array $payload, int $httpCode = 200): void
{
    http_response_code($httpCode);
    echo json_encode($payload, JSON_UNESCAPED_SLASHES);
    exit;
}

if (strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? 'GET')) === 'OPTIONS') {
    api_respond(['ok' => true, 'preflight' => true], 200);
}

function api_require_https(): void
{
    $httpsOn = false;
    if (!empty($_SERVER['HTTPS']) && strtolower((string)$_SERVER['HTTPS']) !== 'off') {
        $httpsOn = true;
    }
    if ((string)($_SERVER['SERVER_PORT'] ?? '') === '443') {
        $httpsOn = true;
    }
    if (strtolower((string)($_SERVER['HTTP_X_FORWARDED_PROTO'] ?? '')) === 'https') {
        $httpsOn = true;
    }

    if (!$httpsOn) {
        api_respond(['ok' => false, 'error' => 'HTTPS_REQUIRED'], 400);
    }
}

function api_require_post(): void
{
    if (strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? 'GET')) !== 'POST') {
        api_respond(['ok' => false, 'error' => 'METHOD_NOT_ALLOWED'], 405);
    }
}

function api_content_type(): string
{
    $contentType = (string)($_SERVER['CONTENT_TYPE'] ?? $_SERVER['HTTP_CONTENT_TYPE'] ?? '');
    $contentType = strtolower(trim($contentType));
    if ($contentType === '') {
        return '';
    }
    $parts = explode(';', $contentType, 2);
    return trim((string)($parts[0] ?? ''));
}

function api_bad_json_response(string $rawBody, string $contentType): void
{
    $payload = ['ok' => false, 'error' => 'BAD_JSON'];
    $debugEnabled = false;
    try {
        $debugEnabled = gmap_debug_api_enabled();
    } catch (Throwable $e) {
        $debugEnabled = false;
    }
    if ($debugEnabled) {
        $preview = substr($rawBody, 0, 200);
        $preview = preg_replace('/[^\P{C}\n\r\t]/u', '', $preview) ?? $preview;
        $payload['debug'] = [
            'content_type' => $contentType,
            'raw_length' => strlen($rawBody),
            'raw_preview' => $preview,
            'json_error' => json_last_error_msg(),
        ];
    }
    api_respond($payload, 400);
}

function api_get_json_body(): array
{
    $raw = file_get_contents('php://input');
    if ($raw === false) {
        $raw = '';
    }

    // Strip UTF-8 BOM if present (common on some shared-hosting relays/tools).
    if (strncmp($raw, "\xEF\xBB\xBF", 3) === 0) {
        $raw = substr($raw, 3);
    }

    $contentType = api_content_type();
    $trimmed = trim($raw);

    if ($trimmed === '') {
        if (!empty($_POST) && is_array($_POST)) {
            return $_POST;
        }
        return [];
    }

    $looksLikeJson = ($trimmed[0] ?? '') === '{' || ($trimmed[0] ?? '') === '[';
    $isJsonType = in_array($contentType, ['application/json', 'text/json'], true);

    if ($isJsonType || $looksLikeJson) {
        $decoded = json_decode($trimmed, true);
        if (!is_array($decoded) || json_last_error() !== JSON_ERROR_NONE) {
            api_bad_json_response($raw, $contentType);
        }
        return $decoded;
    }

    if ($contentType === 'application/x-www-form-urlencoded') {
        parse_str($trimmed, $parsed);
        return is_array($parsed) ? $parsed : [];
    }

    // Fallback: attempt JSON decode first, then parse_str for compatibility.
    $decoded = json_decode($trimmed, true);
    if (is_array($decoded) && json_last_error() === JSON_ERROR_NONE) {
        return $decoded;
    }

    parse_str($trimmed, $parsedFallback);
    if (is_array($parsedFallback) && !empty($parsedFallback)) {
        return $parsedFallback;
    }

    api_bad_json_response($raw, $contentType);
    return [];
}

function api_validate_hwid_hash(?string $value): string
{
    $hwid = strtolower(trim((string)$value));
    if (!preg_match('/^[a-f0-9]{64}$/', $hwid)) {
        api_respond(['ok' => false, 'error' => 'INVALID_HWID'], 200);
    }
    return $hwid;
}

function api_rate_limit(string $bucket, int $maxHits, int $windowSeconds): void
{
    $path = rtrim(sys_get_temp_dir(), DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . 'gmap_rl_' . sha1($bucket) . '.json';
    $now = time();
    $timestamps = [];

    if (is_file($path)) {
        $existing = json_decode((string)file_get_contents($path), true);
        if (is_array($existing)) {
            foreach ($existing as $ts) {
                if (is_int($ts) && ($now - $ts) < $windowSeconds) {
                    $timestamps[] = $ts;
                }
            }
        }
    }

    if (count($timestamps) >= $maxHits) {
        api_respond(['ok' => false, 'error' => 'RATE_LIMITED'], 429);
    }

    $timestamps[] = $now;
    file_put_contents($path, json_encode($timestamps), LOCK_EX);
}

function api_license_error(string $errorCode, int $httpCode = 200, string $message = ''): void
{
    $payload = ['ok' => false, 'error' => $errorCode];
    if ($message !== '') {
        $payload['message'] = $message;
    }
    api_respond($payload, $httpCode);
}

function api_require_debug_key(): void
{
    $debugEnabled = false;
    try {
        $debugEnabled = gmap_debug_api_enabled();
    } catch (Throwable $e) {
        api_respond(['ok' => false, 'error' => 'SERVER_ERROR', 'message' => 'Unable to load debug config.'], 500);
    }

    if (!$debugEnabled) {
        api_respond(['ok' => false, 'error' => 'DEBUG_DISABLED'], 403);
    }

    $expected = '';
    try {
        $expected = gmap_debug_key();
    } catch (Throwable $e) {
        api_respond(['ok' => false, 'error' => 'SERVER_ERROR', 'message' => 'Unable to load debug key.'], 500);
    }
    if ($expected === '') {
        api_respond(['ok' => false, 'error' => 'DEBUG_KEY_NOT_CONFIGURED'], 403);
    }

    $provided = trim((string)($_SERVER['HTTP_X_DEBUG_KEY'] ?? ''));
    if ($provided === '' || !hash_equals($expected, $provided)) {
        api_respond(['ok' => false, 'error' => 'FORBIDDEN'], 403);
    }
}
