<?php
declare(strict_types=1);

require_once __DIR__ . '/common.php';
require_once __DIR__ . '/token.php';

try {
    api_require_https();
    api_require_post();
    api_rate_limit('activate:' . gmap_client_ip(), 60, 60);

    $body = api_get_json_body();
    $licenseKey = gmap_normalize_license_key((string)($body['license_key'] ?? ''));
    $hwidHash = api_validate_hwid_hash((string)($body['hwid_hash'] ?? ''));
    $appVersion = substr(trim((string)($body['app_version'] ?? 'unknown')), 0, 64);

    if ($licenseKey === '') {
        api_license_error('INVALID_KEY');
    }

    $pdo = gmap_pdo();
    $now = gmap_now_utc();

    $pdo->beginTransaction();
    $stmt = $pdo->prepare('SELECT * FROM licenses WHERE license_key = :license_key LIMIT 1 FOR UPDATE');
    $stmt->execute([':license_key' => $licenseKey]);
    $license = $stmt->fetch();

    if (!$license) {
        gmap_insert_log($pdo, null, 'deny', $hwidHash, 'INVALID_KEY v=' . $appVersion);
        $pdo->commit();
        api_license_error('INVALID_KEY');
    }

    $licenseId = (int)$license['id'];
    $status = (string)$license['status'];
    $expiresAt = (string)$license['expires_at'];
    $expiresTs = strtotime($expiresAt . ' UTC');

    if ($status === 'revoked') {
        gmap_insert_log($pdo, $licenseId, 'deny', $hwidHash, 'KEY_REVOKED v=' . $appVersion);
        $pdo->commit();
        api_license_error('KEY_REVOKED');
    }
    if ($status === 'paused') {
        gmap_insert_log($pdo, $licenseId, 'deny', $hwidHash, 'KEY_PAUSED v=' . $appVersion);
        $pdo->commit();
        api_license_error('KEY_PAUSED');
    }
    if ($expiresTs !== false && $expiresTs < time()) {
        gmap_insert_log($pdo, $licenseId, 'deny', $hwidHash, 'KEY_EXPIRED v=' . $appVersion);
        $pdo->commit();
        api_license_error('KEY_EXPIRED');
    }

    $boundHwid = strtolower(trim((string)($license['bound_hwid'] ?? '')));
    if ($boundHwid === '') {
        $bind = $pdo->prepare(
            'UPDATE licenses
             SET bound_hwid = :bound_hwid, bound_at = :bound_at, last_seen_at = :last_seen_at, last_ip = :last_ip
             WHERE id = :id'
        );
        $bind->execute([
            ':bound_hwid' => $hwidHash,
            ':bound_at' => $now,
            ':last_seen_at' => $now,
            ':last_ip' => gmap_client_ip(),
            ':id' => $licenseId,
        ]);
        $license['bound_hwid'] = $hwidHash;
        gmap_insert_log($pdo, $licenseId, 'activate', $hwidHash, 'Bound on first activation v=' . $appVersion);
    } elseif (!hash_equals($boundHwid, $hwidHash)) {
        gmap_insert_log($pdo, $licenseId, 'deny', $hwidHash, 'KEY_IN_USE v=' . $appVersion);
        $pdo->commit();
        api_license_error('KEY_IN_USE');
    } else {
        $touch = $pdo->prepare('UPDATE licenses SET last_seen_at = :last_seen_at, last_ip = :last_ip WHERE id = :id');
        $touch->execute([
            ':last_seen_at' => $now,
            ':last_ip' => gmap_client_ip(),
            ':id' => $licenseId,
        ]);
        gmap_insert_log($pdo, $licenseId, 'activate', $hwidHash, 'Re-activation on bound machine v=' . $appVersion);
    }

    $token = gmap_issue_license_token($license, $hwidHash, gmap_hmac_secret());
    $pdo->commit();

    api_respond([
        'ok' => true,
        'token' => $token,
        'expires_at' => $expiresAt,
        'status' => $status,
    ]);
} catch (Throwable $e) {
    if (isset($pdo) && $pdo instanceof PDO && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    api_respond([
        'ok' => false,
        'error' => 'SERVER_ERROR',
        'message' => 'Activation failed.',
    ], 500);
}
