<?php
declare(strict_types=1);

session_start();
require_once __DIR__ . '/../api/db.php';

function admin_h(string $value): string
{
    return htmlspecialchars($value, ENT_QUOTES, 'UTF-8');
}

function admin_is_logged_in(): bool
{
    return !empty($_SESSION['admin_logged_in']);
}

function admin_require_login(): void
{
    if (!admin_is_logged_in()) {
        header('Location: login.php');
        exit;
    }
}

function admin_csrf_token(): string
{
    if (empty($_SESSION['admin_csrf'])) {
        $_SESSION['admin_csrf'] = bin2hex(random_bytes(32));
    }
    return (string)$_SESSION['admin_csrf'];
}

function admin_verify_csrf(?string $token): bool
{
    $session = (string)($_SESSION['admin_csrf'] ?? '');
    $token = (string)$token;
    if ($session === '' || $token === '') {
        return false;
    }
    return hash_equals($session, $token);
}

function admin_set_flash(string $message): void
{
    $_SESSION['admin_flash'] = $message;
}

function admin_get_flash(): string
{
    $msg = (string)($_SESSION['admin_flash'] ?? '');
    unset($_SESSION['admin_flash']);
    return $msg;
}

function admin_login(string $username, string $password): bool
{
    $username = trim($username);
    if ($username === '' || $password === '') {
        return false;
    }

    $pdo = gmap_pdo();
    $stmt = $pdo->prepare('SELECT username, password_hash FROM admin_users WHERE username = :u AND active = 1 LIMIT 1');
    $stmt->execute([':u' => $username]);
    $row = $stmt->fetch();
    if ($row && password_verify($password, (string)$row['password_hash'])) {
        $_SESSION['admin_logged_in'] = true;
        $_SESSION['admin_username'] = (string)$row['username'];
        return true;
    }

    $cfgUser = gmap_admin_username();
    $cfgHash = gmap_admin_password_hash();
    if ($cfgUser !== '' && $cfgHash !== '' && hash_equals($cfgUser, $username) && password_verify($password, $cfgHash)) {
        $_SESSION['admin_logged_in'] = true;
        $_SESSION['admin_username'] = $cfgUser;
        return true;
    }

    return false;
}
