<?php
declare(strict_types=1);
require_once __DIR__ . '/bootstrap.php';

admin_require_login();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: index.php');
    exit;
}

if (!admin_verify_csrf((string)($_POST['csrf_token'] ?? ''))) {
    admin_set_flash('Invalid CSRF token.');
    header('Location: index.php');
    exit;
}

$action = (string)($_POST['action'] ?? '');

try {
    $pdo = gmap_pdo();

    if ($action === 'create_license') {
        $userLabel = trim((string)($_POST['user_label'] ?? ''));
        $email = trim((string)($_POST['email'] ?? ''));
        $days = max(1, min(3650, (int)($_POST['duration_days'] ?? 30)));
        $notes = trim((string)($_POST['notes'] ?? ''));

        if ($userLabel === '') {
            throw new RuntimeException('User label is required.');
        }

        $licenseKey = gmap_generate_license_key();
        $expiresAt = gmdate('Y-m-d H:i:s', time() + ($days * 86400));

        $stmt = $pdo->prepare(
            'INSERT INTO licenses (license_key, user_label, email, status, created_at, expires_at, notes, max_devices)
             VALUES (:license_key, :user_label, :email, :status, :created_at, :expires_at, :notes, :max_devices)'
        );
        $stmt->execute([
            ':license_key' => $licenseKey,
            ':user_label' => $userLabel,
            ':email' => $email !== '' ? $email : null,
            ':status' => 'active',
            ':created_at' => gmap_now_utc(),
            ':expires_at' => $expiresAt,
            ':notes' => $notes !== '' ? $notes : null,
            ':max_devices' => 1,
        ]);
        $licenseId = (int)$pdo->lastInsertId();
        gmap_insert_log($pdo, $licenseId, 'create', null, 'Created by admin');
        admin_set_flash('License created: ' . gmap_format_license_key($licenseKey));
    } elseif ($action === 'extend_license') {
        $licenseId = (int)($_POST['license_id'] ?? 0);
        $days = max(1, min(3650, (int)($_POST['extend_days'] ?? 30)));
        if ($licenseId <= 0) {
            throw new RuntimeException('Invalid license id.');
        }

        $select = $pdo->prepare('SELECT expires_at FROM licenses WHERE id = :id LIMIT 1');
        $select->execute([':id' => $licenseId]);
        $row = $select->fetch();
        if (!$row) {
            throw new RuntimeException('License not found.');
        }

        $baseTs = strtotime((string)$row['expires_at'] . ' UTC');
        if ($baseTs === false || $baseTs < time()) {
            $baseTs = time();
        }
        $newExpires = gmdate('Y-m-d H:i:s', $baseTs + ($days * 86400));

        $update = $pdo->prepare('UPDATE licenses SET expires_at = :expires_at WHERE id = :id');
        $update->execute([':expires_at' => $newExpires, ':id' => $licenseId]);
        gmap_insert_log($pdo, $licenseId, 'extend', null, 'Extended by ' . $days . ' days');
        admin_set_flash('License expiry extended.');
    } elseif ($action === 'set_status') {
        $licenseId = (int)($_POST['license_id'] ?? 0);
        $status = (string)($_POST['status'] ?? '');
        $allowed = ['active', 'paused', 'revoked'];
        if ($licenseId <= 0 || !in_array($status, $allowed, true)) {
            throw new RuntimeException('Invalid status request.');
        }

        $update = $pdo->prepare('UPDATE licenses SET status = :status WHERE id = :id');
        $update->execute([':status' => $status, ':id' => $licenseId]);

        $event = $status === 'revoked' ? 'revoke' : ($status === 'paused' ? 'pause' : 'resume');
        gmap_insert_log($pdo, $licenseId, $event, null, 'Status changed to ' . $status);
        admin_set_flash('License status updated.');
    } elseif ($action === 'reset_hwid') {
        $licenseId = (int)($_POST['license_id'] ?? 0);
        if ($licenseId <= 0) {
            throw new RuntimeException('Invalid license id.');
        }
        $update = $pdo->prepare('UPDATE licenses SET bound_hwid = NULL, bound_at = NULL WHERE id = :id');
        $update->execute([':id' => $licenseId]);
        gmap_insert_log($pdo, $licenseId, 'reset', null, 'HWID binding reset by admin');
        admin_set_flash('HWID binding reset.');
    } elseif ($action === 'delete_license') {
        $licenseId = (int)($_POST['license_id'] ?? 0);
        if ($licenseId <= 0) {
            throw new RuntimeException('Invalid license id.');
        }
        gmap_insert_log($pdo, $licenseId, 'delete', null, 'License deleted by admin');
        $delete = $pdo->prepare('DELETE FROM licenses WHERE id = :id');
        $delete->execute([':id' => $licenseId]);
        admin_set_flash('License deleted.');
    } else {
        throw new RuntimeException('Unknown action.');
    }
} catch (Throwable $e) {
    admin_set_flash('Action failed: ' . $e->getMessage());
}

header('Location: index.php');
exit;
