<?php
declare(strict_types=1);

require_once __DIR__ . '/db.php';

function gmap_b64url_encode(string $raw): string
{
    return rtrim(strtr(base64_encode($raw), '+/', '-_'), '=');
}

function gmap_b64url_decode(string $raw): string
{
    $pad = strlen($raw) % 4;
    if ($pad > 0) {
        $raw .= str_repeat('=', 4 - $pad);
    }
    return (string)base64_decode(strtr($raw, '-_', '+/'), true);
}

function gmap_sign_token(array $payload, string $secret): string
{
    $header = ['alg' => 'HS256', 'typ' => 'JWT'];
    $headerB64 = gmap_b64url_encode((string)json_encode($header, JSON_UNESCAPED_SLASHES));
    $payloadB64 = gmap_b64url_encode((string)json_encode($payload, JSON_UNESCAPED_SLASHES));
    $signature = hash_hmac('sha256', $headerB64 . '.' . $payloadB64, $secret, true);
    $sigB64 = gmap_b64url_encode($signature);
    return $headerB64 . '.' . $payloadB64 . '.' . $sigB64;
}

function gmap_verify_token(string $token, string $secret): ?array
{
    $parts = explode('.', $token);
    if (count($parts) !== 3) {
        return null;
    }
    [$headerB64, $payloadB64, $sigB64] = $parts;
    $expected = gmap_b64url_encode(hash_hmac('sha256', $headerB64 . '.' . $payloadB64, $secret, true));
    if (!hash_equals($expected, $sigB64)) {
        return null;
    }

    $payloadJson = gmap_b64url_decode($payloadB64);
    $payload = json_decode($payloadJson, true);
    if (!is_array($payload)) {
        return null;
    }
    return $payload;
}

function gmap_issue_license_token(array $licenseRow, string $hwidHash, string $secret): string
{
    $normalizedKey = gmap_normalize_license_key((string)$licenseRow['license_key']);
    $expiresAtTs = strtotime((string)$licenseRow['expires_at'] . ' UTC');
    $payload = [
        'license_id' => (int)$licenseRow['id'],
        'license_key_hash' => gmap_hash_license_key($normalizedKey),
        'hwid_hash' => $hwidHash,
        'issued_at' => time(),
        'expires_at' => $expiresAtTs ?: time(),
        'status' => (string)$licenseRow['status'],
        'features' => ['scrape'],
    ];
    return gmap_sign_token($payload, $secret);
}
