<?php
declare(strict_types=1);

require_once __DIR__ . '/db.php';

header('Content-Type: application/json; charset=utf-8');

function api_respond(array $payload, int $httpCode = 200): void
{
    http_response_code($httpCode);
    echo json_encode($payload, JSON_UNESCAPED_SLASHES);
    exit;
}

function api_require_https(): void
{
    $httpsOn = false;
    if (!empty($_SERVER['HTTPS']) && strtolower((string)$_SERVER['HTTPS']) !== 'off') {
        $httpsOn = true;
    }
    if ((string)($_SERVER['SERVER_PORT'] ?? '') === '443') {
        $httpsOn = true;
    }
    if (strtolower((string)($_SERVER['HTTP_X_FORWARDED_PROTO'] ?? '')) === 'https') {
        $httpsOn = true;
    }

    if (!$httpsOn) {
        api_respond(['ok' => false, 'error' => 'HTTPS_REQUIRED'], 400);
    }
}

function api_require_post(): void
{
    if (strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? 'GET')) !== 'POST') {
        api_respond(['ok' => false, 'error' => 'METHOD_NOT_ALLOWED'], 405);
    }
}

function api_get_json_body(): array
{
    $raw = file_get_contents('php://input');
    if ($raw === false || trim($raw) === '') {
        return [];
    }
    $decoded = json_decode($raw, true);
    if (!is_array($decoded)) {
        api_respond(['ok' => false, 'error' => 'BAD_JSON'], 400);
    }
    return $decoded;
}

function api_validate_hwid_hash(?string $value): string
{
    $hwid = strtolower(trim((string)$value));
    if (!preg_match('/^[a-f0-9]{64}$/', $hwid)) {
        api_respond(['ok' => false, 'error' => 'INVALID_HWID'], 200);
    }
    return $hwid;
}

function api_rate_limit(string $bucket, int $maxHits, int $windowSeconds): void
{
    $path = rtrim(sys_get_temp_dir(), DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . 'gmap_rl_' . sha1($bucket) . '.json';
    $now = time();
    $timestamps = [];

    if (is_file($path)) {
        $existing = json_decode((string)file_get_contents($path), true);
        if (is_array($existing)) {
            foreach ($existing as $ts) {
                if (is_int($ts) && ($now - $ts) < $windowSeconds) {
                    $timestamps[] = $ts;
                }
            }
        }
    }

    if (count($timestamps) >= $maxHits) {
        api_respond(['ok' => false, 'error' => 'RATE_LIMITED'], 429);
    }

    $timestamps[] = $now;
    file_put_contents($path, json_encode($timestamps), LOCK_EX);
}

function api_license_error(string $errorCode, int $httpCode = 200, string $message = ''): void
{
    $payload = ['ok' => false, 'error' => $errorCode];
    if ($message !== '') {
        $payload['message'] = $message;
    }
    api_respond($payload, $httpCode);
}
